import type { ModelInfo as ModelInfoType } from "@shared/api"
import { ANTHROPIC_MIN_THINKING_BUDGET, ApiProvider } from "@shared/api"
import { StringRequest } from "@shared/proto/cline/common"
import { UpdateSettingsRequest } from "@shared/proto/cline/state"
import { Mode } from "@shared/storage/types"
import { ArrowLeftRight, Brain, Check, ChevronDownIcon, Search, Settings } from "lucide-react"
import React, { useCallback, useEffect, useMemo, useRef, useState } from "react"
import { createPortal } from "react-dom"
import { useWindowSize } from "react-use"
import styled from "styled-components"
import { CODE_BLOCK_BG_COLOR } from "@/components/common/CodeBlock"
import PopupModalContainer from "@/components/common/PopupModalContainer"

const PLAN_MODE_COLOR = "var(--vscode-activityWarningBadge-background)"
const ACT_MODE_COLOR = "var(--vscode-focusBorder)"

import { freeModels, recommendedModels } from "@/components/settings/OpenRouterModelPicker"
import { SUPPORTED_ANTHROPIC_THINKING_MODELS } from "@/components/settings/providers/AnthropicProvider"
import { SUPPORTED_BEDROCK_THINKING_MODELS } from "@/components/settings/providers/BedrockProvider"
import {
	filterOpenRouterModelIds,
	getModelsForProvider,
	getModeSpecificFields,
	getProviderInfo,
	normalizeApiConfiguration,
	syncModeConfigurations,
} from "@/components/settings/utils/providerUtils"
import { useApiConfigurationHandlers } from "@/components/settings/utils/useApiConfigurationHandlers"
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip"
import { useExtensionState } from "@/context/ExtensionStateContext"
import { StateServiceClient } from "@/services/grpc-client"
import { getConfiguredProviders, getProviderLabel } from "@/utils/getConfiguredProviders"
import ThinkingBudgetSlider from "../settings/ThinkingBudgetSlider"

const SETTINGS_ONLY_PROVIDERS: ApiProvider[] = [
	"openai",
	"ollama",
	"lmstudio",
	"vscode-lm",
	"requesty",
	"hicap",
	"dify",
	"oca",
	"aihubmix",
	"together",
]

const OPENROUTER_MODEL_PROVIDERS: ApiProvider[] = ["cline", "openrouter", "vercel-ai-gateway"]

interface ModelPickerModalProps {
	isOpen: boolean
	onOpenChange: (open: boolean) => void
	currentMode: Mode
	children: React.ReactNode
}

interface ModelItem {
	id: string
	name: string
	provider?: string
	description?: string
	label?: string
	info?: ModelInfoType
}

// Star icon for favorites (only for openrouter/vercel-ai-gateway providers)
const StarIcon = ({ isFavorite, onClick }: { isFavorite: boolean; onClick: (e: React.MouseEvent) => void }) => {
	return (
		<div
			onClick={onClick}
			style={{
				cursor: "pointer",
				color: isFavorite ? "var(--vscode-terminal-ansiYellow)" : "var(--vscode-descriptionForeground)",
				marginLeft: "8px",
				fontSize: "14px",
				display: "flex",
				alignItems: "center",
				justifyContent: "center",
				userSelect: "none",
				WebkitUserSelect: "none",
			}}>
			{isFavorite ? "★" : "☆"}
		</div>
	)
}

const ModelPickerModal: React.FC<ModelPickerModalProps> = ({ isOpen, onOpenChange, currentMode, children }) => {
	const {
		apiConfiguration,
		openRouterModels,
		vercelAiGatewayModels,
		navigateToSettings,
		planActSeparateModelsSetting,
		showSettings,
		showMcp,
		showHistory,
		showAccount,
		remoteConfigSettings,
		favoritedModelIds,
		basetenModels,
		liteLlmModels,
	} = useExtensionState()
	const { handleModeFieldChange, handleModeFieldsChange, handleFieldsChange } = useApiConfigurationHandlers()

	const [searchQuery, setSearchQuery] = useState("")
	const [activeEditMode, setActiveEditMode] = useState<Mode>(currentMode) // which mode we're editing in split view
	const [menuPosition, setMenuPosition] = useState(0)
	const [arrowPosition, setArrowPosition] = useState(0)
	const [isProviderExpanded, setIsProviderExpanded] = useState(false)
	const [providerDropdownPosition, setProviderDropdownPosition] = useState({ top: 0, left: 0, width: 0, maxHeight: 200 })
	const [selectedIndex, setSelectedIndex] = useState(-1) // For keyboard navigation
	const searchInputRef = useRef<HTMLInputElement>(null)
	const triggerRef = useRef<HTMLDivElement>(null)
	const modalRef = useRef<HTMLDivElement>(null)
	const providerRowRef = useRef<HTMLDivElement>(null)
	const providerDropdownRef = useRef<HTMLDivElement>(null)
	const itemRefs = useRef<(HTMLDivElement | null)[]>([]) // For scrollIntoView
	const { width: viewportWidth, height: viewportHeight } = useWindowSize()

	// Get current provider from config - use activeEditMode when in split mode
	const effectiveMode = planActSeparateModelsSetting ? activeEditMode : currentMode
	const { selectedProvider, selectedModelId, selectedModelInfo } = normalizeApiConfiguration(apiConfiguration, effectiveMode)

	// Get both Plan and Act models for split view
	const planModel = useMemo(() => normalizeApiConfiguration(apiConfiguration, "plan"), [apiConfiguration])
	const actModel = useMemo(() => normalizeApiConfiguration(apiConfiguration, "act"), [apiConfiguration])

	// Use the setting for split mode
	const isSplit = planActSeparateModelsSetting

	// Check if model supports thinking (token-based budget)
	// OpenAI Codex uses discrete reasoning effort levels controlled via global settings, not token budgets
	const supportsThinking = useMemo(() => {
		if (selectedProvider === "openai-codex") {
			return false
		}
		if (selectedProvider === "anthropic" || selectedProvider === "claude-code") {
			return SUPPORTED_ANTHROPIC_THINKING_MODELS.includes(selectedModelId)
		}
		if (selectedProvider === "bedrock") {
			return SUPPORTED_BEDROCK_THINKING_MODELS.includes(selectedModelId)
		}
		return selectedModelInfo?.supportsReasoning || !!selectedModelInfo?.thinkingConfig
	}, [selectedProvider, selectedModelId, selectedModelInfo])

	// Get thinking budget from current mode config
	const modeFields = getModeSpecificFields(apiConfiguration, currentMode)
	const thinkingBudget = modeFields.thinkingBudgetTokens || 0
	const thinkingEnabled = thinkingBudget > 0

	// Handle thinking toggle - uses ANTHROPIC_MIN_THINKING_BUDGET as default when enabling
	const handleThinkingToggle = useCallback(
		(enabled: boolean) => {
			const budget = enabled ? ANTHROPIC_MIN_THINKING_BUDGET : 0
			handleModeFieldChange(
				{ plan: "planModeThinkingBudgetTokens", act: "actModeThinkingBudgetTokens" },
				budget,
				currentMode,
			)
		},
		[handleModeFieldChange, currentMode],
	)

	// Get configured providers
	const configuredProviders = useMemo(() => {
		if (remoteConfigSettings?.remoteConfiguredProviders?.length) {
			return remoteConfigSettings.remoteConfiguredProviders
		}

		return getConfiguredProviders(apiConfiguration)
	}, [apiConfiguration, remoteConfigSettings?.remoteConfiguredProviders])

	// Get models for current provider
	const allModels = useMemo((): ModelItem[] => {
		if (OPENROUTER_MODEL_PROVIDERS.includes(selectedProvider)) {
			// Use vercelAiGatewayModels for Vercel provider, openRouterModels for others
			const modelsSource = selectedProvider === "vercel-ai-gateway" ? vercelAiGatewayModels : openRouterModels
			const modelIds = Object.keys(modelsSource || {})
			const filteredIds = filterOpenRouterModelIds(modelIds, selectedProvider)

			return filteredIds.map((id) => ({
				id,
				name: id.split("/").pop() || id,
				provider: id.split("/")[0],
				info: modelsSource[id],
			}))
		}

		// Use centralized helper for static provider models
		const models = getModelsForProvider(selectedProvider, apiConfiguration, {
			basetenModels,
			liteLlmModels,
		})
		if (models) {
			return Object.entries(models).map(([id, info]) => ({
				id,
				name: id,
				provider: selectedProvider,
				info,
			}))
		}

		return []
	}, [selectedProvider, openRouterModels, vercelAiGatewayModels, apiConfiguration, basetenModels, liteLlmModels])

	// Multi-word substring search - all words must match somewhere in id/name/provider
	const matchesSearch = useCallback((model: ModelItem, query: string): boolean => {
		if (!query.trim()) {
			return true
		}
		const queryWords = query.toLowerCase().trim().split(/\s+/)
		const searchText = `${model.id} ${model.name} ${model.provider || ""}`.toLowerCase()
		return queryWords.every((word) => searchText.includes(word))
	}, [])

	// Filtered models - for OpenRouter/Vercel show all by default, for Cline only when searching
	const filteredModels = useMemo(() => {
		const isCline = selectedProvider === "cline"

		// For Cline: only show non-featured models when searching
		if (isCline && !searchQuery) {
			return []
		}

		let models: ModelItem[]
		if (searchQuery) {
			models = allModels.filter((m) => matchesSearch(m, searchQuery))
		} else {
			// For non-Cline OpenRouter providers: show all models by default
			models = [...allModels]
		}

		// Filter out current model
		models = models.filter((m) => m.id !== selectedModelId)

		// For Cline when searching, also filter out featured models (they're shown separately)
		if (isCline) {
			const featuredIds = new Set([...recommendedModels, ...freeModels].map((m) => m.id))
			models = models.filter((m) => !featuredIds.has(m.id))
		}

		// For openrouter/vercel-ai-gateway (not cline): put favorites first
		if (!isCline && (selectedProvider === "openrouter" || selectedProvider === "vercel-ai-gateway")) {
			const favoriteSet = new Set(favoritedModelIds || [])
			const favoritedModels = models.filter((m) => favoriteSet.has(m.id))
			const nonFavoritedModels = models.filter((m) => !favoriteSet.has(m.id))
			// Sort non-favorited alphabetically by provider
			nonFavoritedModels.sort((a, b) => (a.provider || "").localeCompare(b.provider || ""))
			return [...favoritedModels, ...nonFavoritedModels]
		}

		// Sort alphabetically by provider
		models = models.sort((a, b) => (a.provider || "").localeCompare(b.provider || ""))
		return models
	}, [searchQuery, matchesSearch, selectedModelId, selectedProvider, allModels, favoritedModelIds])

	// Featured models for Cline provider (recommended + free)
	const featuredModels = useMemo(() => {
		if (selectedProvider !== "cline") {
			return []
		}

		const allFeatured = [...recommendedModels, ...freeModels].map((m) => ({
			...m,
			name: m.id.split("/").pop() || m.id,
			provider: m.id.split("/")[0],
		}))

		// Filter out current model
		const filtered = allFeatured.filter((m) => m.id !== selectedModelId)

		// Apply search filter if searching (uses same multi-word logic)
		if (searchQuery) {
			return filtered.filter((m) => matchesSearch(m, searchQuery))
		}

		return filtered
	}, [selectedProvider, searchQuery, selectedModelId, matchesSearch])

	// Handle model selection - in split mode uses activeEditMode, otherwise closes modal
	const handleSelectModel = useCallback(
		(modelId: string, modelInfo?: ModelInfoType) => {
			const modeToUse = isSplit ? activeEditMode : currentMode

			if (selectedProvider === "vercel-ai-gateway") {
				// Vercel AI Gateway uses its own model fields
				const modelInfoToUse = modelInfo || vercelAiGatewayModels[modelId]
				handleModeFieldsChange(
					{
						vercelAiGatewayModelId: { plan: "planModeVercelAiGatewayModelId", act: "actModeVercelAiGatewayModelId" },
						vercelAiGatewayModelInfo: {
							plan: "planModeVercelAiGatewayModelInfo",
							act: "actModeVercelAiGatewayModelInfo",
						},
					},
					{
						vercelAiGatewayModelId: modelId,
						vercelAiGatewayModelInfo: modelInfoToUse,
					},
					modeToUse,
				)
			} else if (OPENROUTER_MODEL_PROVIDERS.includes(selectedProvider)) {
				// Cline and OpenRouter use openRouter fields
				const modelInfoToUse = modelInfo || openRouterModels[modelId]
				handleModeFieldsChange(
					{
						openRouterModelId: { plan: "planModeOpenRouterModelId", act: "actModeOpenRouterModelId" },
						openRouterModelInfo: { plan: "planModeOpenRouterModelInfo", act: "actModeOpenRouterModelInfo" },
					},
					{
						openRouterModelId: modelId,
						openRouterModelInfo: modelInfoToUse,
					},
					modeToUse,
				)
			} else if (selectedProvider === "baseten") {
				// Baseten uses provider-specific model ID and info fields
				handleModeFieldsChange(
					{
						basetenModelId: { plan: "planModeBasetenModelId", act: "actModeBasetenModelId" },
						basetenModelInfo: { plan: "planModeBasetenModelInfo", act: "actModeBasetenModelInfo" },
					},
					{
						basetenModelId: modelId,
						basetenModelInfo: modelInfo,
					},
					modeToUse,
				)
			} else if (selectedProvider === "litellm") {
				// LiteLLM uses provider-specific model ID and info fields
				handleModeFieldsChange(
					{
						liteLlmModelId: { plan: "planModeLiteLlmModelId", act: "actModeLiteLlmModelId" },
						liteLlmModelInfo: { plan: "planModeLiteLlmModelInfo", act: "actModeLiteLlmModelInfo" },
					},
					{
						liteLlmModelId: modelId,
						liteLlmModelInfo: modelInfo,
					},
					modeToUse,
				)
			} else {
				// Static model providers use apiModelId
				handleModeFieldChange({ plan: "planModeApiModelId", act: "actModeApiModelId" }, modelId, modeToUse)
			}
			// Only close modal if not in split mode
			if (!isSplit) {
				onOpenChange(false)
			}
		},
		[
			selectedProvider,
			handleModeFieldsChange,
			handleModeFieldChange,
			currentMode,
			isSplit,
			activeEditMode,
			openRouterModels,
			vercelAiGatewayModels,
			onOpenChange,
		],
	)

	// Handle provider selection from inline list
	const handleProviderSelect = useCallback(
		(provider: ApiProvider) => {
			const modeToUse = isSplit ? activeEditMode : currentMode
			handleModeFieldChange({ plan: "planModeApiProvider", act: "actModeApiProvider" }, provider, modeToUse)
			setIsProviderExpanded(false)
		},
		[handleModeFieldChange, currentMode, isSplit, activeEditMode],
	)

	// Handle split toggle - should NOT close modal
	const handleSplitToggle = useCallback(
		async (enabled: boolean) => {
			// Update the setting
			await StateServiceClient.updateSettings(
				UpdateSettingsRequest.create({
					planActSeparateModelsSetting: enabled,
				}),
			)
			// If disabling split mode, sync configurations
			if (!enabled) {
				syncModeConfigurations(apiConfiguration, currentMode, handleFieldsChange)
			}
		},
		[apiConfiguration, currentMode, handleFieldsChange],
	)

	// Handle configure link click
	const handleConfigureClick = useCallback(
		(e: React.MouseEvent) => {
			e.stopPropagation()
			e.preventDefault()
			onOpenChange(false)
			navigateToSettings?.()
		},
		[onOpenChange, navigateToSettings],
	)

	// Keyboard navigation handler
	const handleKeyDown = useCallback(
		(e: React.KeyboardEvent<HTMLInputElement>) => {
			const totalItems = filteredModels.length + featuredModels.length
			if (totalItems === 0) {
				return
			}

			switch (e.key) {
				case "ArrowDown":
					e.preventDefault()
					setSelectedIndex((prev) => (prev < totalItems - 1 ? prev + 1 : prev))
					break
				case "ArrowUp":
					e.preventDefault()
					setSelectedIndex((prev) => (prev > 0 ? prev - 1 : prev))
					break
				case "Enter":
					e.preventDefault()
					if (selectedIndex >= 0) {
						// Determine which list the index falls into
						if (selectedIndex < featuredModels.length) {
							const model = featuredModels[selectedIndex]
							handleSelectModel(model.id, openRouterModels[model.id])
						} else {
							const model = filteredModels[selectedIndex - featuredModels.length]
							handleSelectModel(model.id, model.info)
						}
					}
					break
				case "Escape":
					e.preventDefault()
					onOpenChange(false)
					break
			}
		},
		[filteredModels, featuredModels, selectedIndex, handleSelectModel, openRouterModels, onOpenChange],
	)

	// Reset selectedIndex and clear refs when search/provider changes
	useEffect(() => {
		setSelectedIndex(-1)
		itemRefs.current = []
	}, [searchQuery, selectedProvider])

	// Scroll selected item into view
	useEffect(() => {
		if (selectedIndex >= 0) {
			// Use requestAnimationFrame to ensure DOM is updated
			requestAnimationFrame(() => {
				const element = itemRefs.current[selectedIndex]
				if (element) {
					element.scrollIntoView({
						block: "nearest",
						behavior: "smooth",
					})
				}
			})
		}
	}, [selectedIndex])

	// Reset states when opening/closing
	useEffect(() => {
		if (isOpen) {
			setIsProviderExpanded(false)
			setSelectedIndex(-1)
			setTimeout(() => searchInputRef.current?.focus(), 100)
		} else {
			setSearchQuery("")
			setSelectedIndex(-1)
		}
	}, [isOpen])

	// Calculate positions for modal and arrow (update on viewport resize)
	useEffect(() => {
		if (isOpen && triggerRef.current) {
			const rect = triggerRef.current.getBoundingClientRect()
			const buttonCenter = rect.left + rect.width / 2
			const rightPosition = document.documentElement.clientWidth - buttonCenter - 5
			setMenuPosition(rect.top + 1)
			setArrowPosition(rightPosition)
		}
	}, [isOpen, viewportWidth, viewportHeight])

	// Handle click outside to close
	useEffect(() => {
		if (!isOpen) {
			return
		}

		const handleClickOutside = (e: MouseEvent) => {
			// Don't close if clicking inside modal, trigger, or provider dropdown portal
			if (
				modalRef.current &&
				!modalRef.current.contains(e.target as Node) &&
				triggerRef.current &&
				!triggerRef.current.contains(e.target as Node) &&
				(!providerDropdownRef.current || !providerDropdownRef.current.contains(e.target as Node))
			) {
				onOpenChange(false)
			}
		}

		// Delay adding listener to avoid immediate close
		const timeoutId = setTimeout(() => {
			document.addEventListener("mousedown", handleClickOutside)
		}, 0)

		return () => {
			clearTimeout(timeoutId)
			document.removeEventListener("mousedown", handleClickOutside)
		}
	}, [isOpen, onOpenChange])

	// Handle escape key
	useEffect(() => {
		if (!isOpen) {
			return
		}

		const handleEscape = (e: KeyboardEvent) => {
			if (e.key === "Escape") {
				onOpenChange(false)
			}
		}

		document.addEventListener("keydown", handleEscape)
		return () => document.removeEventListener("keydown", handleEscape)
	}, [isOpen, onOpenChange])

	// Close modal when navigating to other views (settings, MCP, history, account)
	useEffect(() => {
		if (isOpen && (showSettings || showMcp || showHistory || showAccount)) {
			onOpenChange(false)
		}
	}, [isOpen, showSettings, showMcp, showHistory, showAccount, onOpenChange])

	// Check if current model actually belongs to current provider (not auto-selected fallback)
	const modelBelongsToProvider = useMemo(() => {
		if (!selectedModelId) {
			return false
		}
		return allModels.some((m) => m.id === selectedModelId)
	}, [selectedModelId, allModels])

	// Handle trigger click
	const handleTriggerClick = useCallback(() => {
		onOpenChange(!isOpen)
	}, [isOpen, onOpenChange])

	const isClineProvider = selectedProvider === "cline"
	const isSearching = !!searchQuery

	return (
		<>
			{/* Trigger wrapper */}
			<div onClick={handleTriggerClick} ref={triggerRef} style={{ cursor: "pointer", display: "inline", minWidth: 0 }}>
				{children}
			</div>

			{/* Modal - rendered via portal with fixed positioning */}
			{isOpen &&
				createPortal(
					<PopupModalContainer
						$arrowPosition={arrowPosition}
						$bottomOffset={5}
						$maxHeight="18em"
						$menuPosition={menuPosition}
						ref={modalRef}>
						{/* Search */}
						<SearchContainer>
							<Search size={14} style={{ color: "var(--vscode-descriptionForeground)", flexShrink: 0 }} />
							<SearchInput
								onChange={(e) => {
									setSearchQuery(e.target.value)
									setIsProviderExpanded(false)
								}}
								onKeyDown={handleKeyDown}
								placeholder={`Search ${allModels.length} models`}
								ref={searchInputRef as any}
								value={searchQuery}
							/>
						</SearchContainer>

						{/* Settings section - provider + icon toggles */}
						<SettingsSection onClick={(e) => e.stopPropagation()}>
							<div className="flex items-center justify-between">
								{/* Provider - collapsible with dropdown portal */}
								<Tooltip>
									<TooltipTrigger asChild>
										<ProviderRow
											onClick={() => {
												if (providerRowRef.current) {
													const rect = providerRowRef.current.getBoundingClientRect()
													const viewportHeight = window.innerHeight
													const spaceBelow = viewportHeight - rect.bottom
													const itemHeight = 28 // approximate height per item
													const numItems = configuredProviders.length + 1 // +1 for "Add provider"
													const dropdownHeight = Math.min(numItems * itemHeight + 8, 200) // 8px for padding

													// If not enough space below, position above
													const shouldFlipUp = spaceBelow < dropdownHeight + 10 && rect.top > spaceBelow

													setProviderDropdownPosition({
														top: shouldFlipUp ? rect.top - dropdownHeight - 4 : rect.bottom + 4,
														left: rect.left,
														width: modalRef.current?.getBoundingClientRect().width || rect.width,
														maxHeight: shouldFlipUp ? rect.top - 10 : spaceBelow - 10,
													})
												}
												setIsProviderExpanded(!isProviderExpanded)
											}}
											ref={providerRowRef}>
											<div className="text-[11px] text-description">Provider:</div>
											<span className="text-[11px] text-description">
												{getProviderLabel(selectedProvider)}
											</span>
											<ChevronDownIcon className="text-description" size={12} />
										</ProviderRow>
									</TooltipTrigger>
									{!isProviderExpanded && (
										<TooltipContent side="top" style={{ zIndex: 9999 }}>
											Configured providers
										</TooltipContent>
									)}
								</Tooltip>

								{/* Icon toggles */}
								<div className="flex items-center gap-2">
									<Tooltip>
										<TooltipTrigger asChild>
											<IconToggle
												$isActive={thinkingEnabled}
												$isHidden={!supportsThinking}
												onClick={(e) => {
													e.stopPropagation()
													supportsThinking && handleThinkingToggle(!thinkingEnabled)
												}}>
												<Brain size={14} />
											</IconToggle>
										</TooltipTrigger>
										{supportsThinking && (
											<TooltipContent side="top" style={{ zIndex: 9999 }}>
												{thinkingEnabled
													? "Extended thinking enabled"
													: "Enable extended thinking for enhanced reasoning"}
											</TooltipContent>
										)}
									</Tooltip>
									<Tooltip>
										<TooltipTrigger asChild>
											<IconToggle
												$isActive={isSplit}
												onClick={(e) => {
													e.stopPropagation()
													handleSplitToggle(!isSplit)
												}}>
												<ArrowLeftRight size={14} />
											</IconToggle>
										</TooltipTrigger>
										<TooltipContent side="top" style={{ zIndex: 9999 }}>
											Use different models for Plan vs Act
										</TooltipContent>
									</Tooltip>
								</div>
							</div>
							{/* Thinking budget slider - shown when model supports thinking, greyed out when disabled */}
							{supportsThinking && (
								<div className="flex items-center gap-2 py-1.5 px-0 mt-0.5 w-full">
									<div className="text-description whitespace-nowrap min-w-[130px] text-[10px]">
										Thinking ({(thinkingBudget ?? 0).toLocaleString()} tokens)
									</div>
									<ThinkingBudgetSlider currentMode={currentMode} showEnableToggle={false} />
								</div>
							)}
						</SettingsSection>

						{/* Scrollable content */}
						<ModelListContainer>
							{/* Current model - inside scroll area for seamless scrolling */}
							{isSplit ? (
								<SplitModeRow onClick={(e) => e.stopPropagation()}>
									<Tooltip>
										<TooltipTrigger asChild>
											<SplitModeCell
												$isActive={activeEditMode === "plan"}
												onClick={() => setActiveEditMode("plan")}>
												<SplitModeLabel $mode="plan">P</SplitModeLabel>
												<SplitModeModel>
													{planModel.selectedModelId?.split("/").pop() || "Not set"}
												</SplitModeModel>
											</SplitModeCell>
										</TooltipTrigger>
										<TooltipContent side="top" style={{ zIndex: 9999 }}>
											Plan mode
										</TooltipContent>
									</Tooltip>
									<Tooltip>
										<TooltipTrigger asChild>
											<SplitModeCell
												$isActive={activeEditMode === "act"}
												onClick={() => setActiveEditMode("act")}>
												<SplitModeLabel $mode="act">A</SplitModeLabel>
												<SplitModeModel>
													{actModel.selectedModelId?.split("/").pop() || "Not set"}
												</SplitModeModel>
											</SplitModeCell>
										</TooltipTrigger>
										<TooltipContent side="top" style={{ zIndex: 9999 }}>
											Act mode
										</TooltipContent>
									</Tooltip>
								</SplitModeRow>
							) : selectedModelId && modelBelongsToProvider ? (
								(() => {
									// Check if current model has a featured label (only for Cline provider)
									const currentFeaturedModel = isClineProvider
										? [...recommendedModels, ...freeModels].find((m) => m.id === selectedModelId)
										: undefined
									return (
										<CurrentModelRow onClick={() => onOpenChange(false)}>
											<ModelInfoRow>
												<div className="text-[11px] text-foreground whitespace-nowrap overflow-hidden text-ellipsis">
													{selectedModelId.split("/").pop() || selectedModelId}
												</div>
												<ModelProvider>
													{OPENROUTER_MODEL_PROVIDERS.includes(selectedProvider)
														? selectedModelId.split("/")[0]
														: selectedProvider}
												</ModelProvider>
											</ModelInfoRow>
											{currentFeaturedModel?.label && <ModelLabel>{currentFeaturedModel.label}</ModelLabel>}
											<Check
												size={14}
												style={{
													color: "var(--vscode-foreground)",
													flexShrink: 0,
												}}
											/>
										</CurrentModelRow>
									)
								})()
							) : !selectedModelId && selectedProvider === "vercel-ai-gateway" ? (
								<EmptyModelRow>
									<span className="text-[11px] text-description">Select a model below</span>
								</EmptyModelRow>
							) : null}

							{/* For Cline: Show recommended models */}
							{isClineProvider &&
								featuredModels.map((model, index) => (
									<ModelItemContainer
										$isSelected={index === selectedIndex}
										key={model.id}
										onClick={() => handleSelectModel(model.id, openRouterModels[model.id])}
										onMouseEnter={() => setSelectedIndex(index)}
										ref={(el) => (itemRefs.current[index] = el)}>
										<ModelInfoRow>
											<ModelName>{model.name}</ModelName>
											<ModelProvider>{model.provider}</ModelProvider>
										</ModelInfoRow>
										<ModelLabel>{model.label}</ModelLabel>
									</ModelItemContainer>
								))}

							{/* All other models (for non-Cline always, for Cline only when searching) */}
							{filteredModels.map((model, index) => {
								const globalIndex = featuredModels.length + index
								const isFavorite = (favoritedModelIds || []).includes(model.id)
								const showStar = selectedProvider === "openrouter" || selectedProvider === "vercel-ai-gateway"
								return (
									<ModelItemContainer
										$isSelected={globalIndex === selectedIndex}
										key={model.id}
										onClick={() => handleSelectModel(model.id, model.info)}
										onMouseEnter={() => setSelectedIndex(globalIndex)}
										ref={(el) => (itemRefs.current[globalIndex] = el)}>
										<ModelInfoRow>
											<ModelName>{model.name}</ModelName>
											<ModelProvider>{model.provider}</ModelProvider>
										</ModelInfoRow>
										{showStar && (
											<StarIcon
												isFavorite={isFavorite}
												onClick={(e) => {
													e.stopPropagation()
													StateServiceClient.toggleFavoriteModel(
														StringRequest.create({ value: model.id }),
													).catch((error: Error) =>
														console.error("Failed to toggle favorite model:", error),
													)
												}}
											/>
										)}
									</ModelItemContainer>
								)
							})}

							{/* Settings-only providers: show configured model info and help text */}
							{SETTINGS_ONLY_PROVIDERS.includes(selectedProvider) &&
								(() => {
									const providerInfo = getProviderInfo(selectedProvider, apiConfiguration, effectiveMode)
									return (
										<SettingsOnlyContainer>
											{/* Show configured model if exists */}
											{providerInfo.modelId && (
												<div className="flex items-center gap-1.5">
													<div className="text-[10px] text-description shrink-0">Current model:</div>
													<ConfiguredModelName>{providerInfo.modelId}</ConfiguredModelName>
												</div>
											)}
											{/* Show base URL if configured */}
											{providerInfo.baseUrl && (
												<div className="flex items-center gap-1.5">
													<div className="text-[10px] text-description shrink-0">Endpoint:</div>
													<div className="text-[10px] text-description whitespace-nowrap overflow-hidden text-ellipsis font-editor">
														{providerInfo.baseUrl}
													</div>
												</div>
											)}
											{/* Help text / empty state guidance */}
											{!providerInfo.modelId && (
												<div className="text-center text-[11px] text-description py-1 px-0">
													{providerInfo.helpText}
												</div>
											)}
											{/* Configure link */}
											<SettingsOnlyLink onClick={handleConfigureClick}>
												<Settings size={12} />
												<span>{providerInfo.modelId ? "Edit in settings" : "Configure in settings"}</span>
											</SettingsOnlyLink>
										</SettingsOnlyContainer>
									)
								})()}

							{/* Empty state */}
							{isSearching &&
								filteredModels.length === 0 &&
								featuredModels.length === 0 &&
								!SETTINGS_ONLY_PROVIDERS.includes(selectedProvider) && <EmptyState>No models found</EmptyState>}
						</ModelListContainer>
					</PopupModalContainer>,
					document.body,
				)}

			{/* Provider dropdown - rendered via portal to avoid clipping */}
			{isOpen &&
				isProviderExpanded &&
				createPortal(
					<ProviderDropdownPortal
						onClick={(e) => e.stopPropagation()}
						ref={providerDropdownRef}
						style={{
							top: providerDropdownPosition.top,
							left: providerDropdownPosition.left,
							width: providerDropdownPosition.width - 20, // Account for modal padding
							maxHeight: providerDropdownPosition.maxHeight,
						}}>
						{configuredProviders.map((provider) => (
							<ProviderDropdownItem
								$isSelected={provider === selectedProvider}
								key={provider}
								onClick={() => handleProviderSelect(provider)}>
								{provider === selectedProvider && <span style={{ marginRight: 4 }}>✓</span>}
								<span>{getProviderLabel(provider)}</span>
							</ProviderDropdownItem>
						))}
						<ProviderDropdownItem $isSelected={false} onClick={handleConfigureClick}>
							<span style={{ color: "var(--vscode-textLink-foreground)" }}>+ Add provider</span>
						</ProviderDropdownItem>
					</ProviderDropdownPortal>,
					document.body,
				)}
		</>
	)
}

const SearchContainer = styled.div`
	padding: 4px 10px;
	min-height: 28px;
	box-sizing: border-box;
	border-bottom: 1px solid var(--vscode-editorGroup-border);
	display: flex;
	align-items: center;
	gap: 8px;
`

const SearchInput = styled.input`
	flex: 1;
	background: transparent;
	border: none;
	outline: none;
	font-size: 11px;
	color: var(--vscode-foreground);
	&:focus {
		outline: none;
	}
	&::placeholder {
		color: var(--vscode-descriptionForeground);
		opacity: 0.7;
	}
`

const SettingsSection = styled.div`
	position: relative;
	padding: 4px 10px;
	border-bottom: 1px solid var(--vscode-editorGroup-border);
	display: flex;
	flex-direction: column;
`

const IconToggle = styled.button<{ $isActive: boolean; $isDisabled?: boolean; $isHidden?: boolean }>`
	display: ${(props) => (props.$isHidden ? "none" : "flex")};
	align-items: center;
	justify-content: center;
	width: 24px;
	height: 24px;
	background: transparent;
	border: none;
	border-radius: 4px;
	cursor: ${(props) => (props.$isDisabled ? "not-allowed" : "pointer")};
	color: ${(props) =>
		props.$isDisabled
			? "var(--vscode-disabledForeground)"
			: props.$isActive
				? "var(--vscode-textLink-foreground)"
				: "var(--vscode-descriptionForeground)"};
	opacity: ${(props) => (props.$isDisabled ? 0.4 : 1)};
	transition: all 0.15s ease;
	&:hover {
		background: ${(props) => (props.$isDisabled ? "transparent" : "var(--vscode-list-hoverBackground)")};
	}
`

// Inline provider row (clickable to expand)
const ProviderRow = styled.div`
	display: flex;
	align-items: center;
	gap: 6px;
	cursor: pointer;
	&:hover {
		opacity: 0.8;
	}
`

// Provider dropdown rendered via portal to avoid clipping
const ProviderDropdownPortal = styled.div`
	position: fixed;
	display: flex;
	flex-direction: column;
	padding: 4px 0;
	background: ${CODE_BLOCK_BG_COLOR};
	border: 1px solid var(--vscode-editorGroup-border);
	border-radius: 4px;
	max-height: 200px;
	overflow-y: auto;
	z-index: 2000;
	box-shadow: 0 4px 12px rgba(0, 0, 0, 0.4);
`

const ProviderDropdownItem = styled.div<{ $isSelected: boolean }>`
	display: flex;
	align-items: center;
	padding: 4px 8px;
	cursor: pointer;
	font-size: 11px;
	color: ${(props) => (props.$isSelected ? "var(--vscode-foreground)" : "var(--vscode-descriptionForeground)")};
	border-radius: 3px;
	&:hover {
		background: var(--vscode-list-hoverBackground);
	}
`

const ModelListContainer = styled.div`
	flex: 1;
	overflow-y: auto;
	min-height: 0;
	scrollbar-width: thin;
	&::-webkit-scrollbar {
		width: 6px;
	}
	&::-webkit-scrollbar-thumb {
		background: transparent;
		border-radius: 3px;
	}
	&:hover::-webkit-scrollbar-thumb {
		background: var(--vscode-scrollbarSlider-background);
	}
`

const ModelItemContainer = styled.div<{ $isSelected: boolean }>`
	display: flex;
	align-items: center;
	justify-content: space-between;
	padding: 4px 10px;
	min-height: 28px;
	box-sizing: border-box;
	cursor: pointer;
	background: ${(props) => (props.$isSelected ? "var(--vscode-list-activeSelectionBackground)" : "transparent")};
	&:hover {
		background: var(--vscode-list-hoverBackground);
	}
`

const ModelInfoRow = styled.div`
	display: flex;
	align-items: center;
	gap: 6px;
	flex: 1;
	min-width: 0;
`

const ModelName = styled.span`
	font-size: 11px;
	color: var(--vscode-foreground);
	white-space: nowrap;
	overflow: hidden;
	text-overflow: ellipsis;
`

const ModelProvider = styled.span`
	font-size: 10px;
	color: var(--vscode-descriptionForeground);
	white-space: nowrap;
	@media (max-width: 280px) {
		display: none;
	}
`

const ModelLabel = styled.span`
	font-size: 9px;
	color: var(--vscode-textLink-foreground);
	text-transform: uppercase;
	letter-spacing: 0.5px;
	font-weight: 500;
	margin-left: auto;
	margin-right: 8px;
`

const EmptyState = styled.div`
	padding: 12px 10px;
	text-align: center;
	font-size: 11px;
	color: var(--vscode-descriptionForeground);
`

// Empty model row - shown when no model is selected for providers like Vercel
const EmptyModelRow = styled.div`
	display: flex;
	align-items: center;
	justify-content: center;
	padding: 8px 10px;
	min-height: 28px;
	box-sizing: border-box;
	background: ${CODE_BLOCK_BG_COLOR};
	position: sticky;
	top: 0;
	z-index: 1;
	border-bottom: 1px solid var(--vscode-editorGroup-border);
`

// Current model row - highlighted, sticky at top when scrolling, clickable to close
const CurrentModelRow = styled.div`
	display: flex;
	align-items: center;
	justify-content: space-between;
	gap: 6px;
	padding: 4px 10px;
	min-height: 28px;
	box-sizing: border-box;
	cursor: pointer;
	background: linear-gradient(var(--vscode-list-activeSelectionBackground), var(--vscode-list-activeSelectionBackground)),
		${CODE_BLOCK_BG_COLOR};
	position: sticky;
	top: 0;
	z-index: 1;
`

// Split mode components - sticky at top when scrolling
const SplitModeRow = styled.div`
	display: flex;
	align-items: stretch;
	gap: 0;
	position: sticky;
	top: 0;
	z-index: 1;
	background: ${CODE_BLOCK_BG_COLOR};
`

const SplitModeCell = styled.div<{ $isActive: boolean }>`
	display: flex;
	align-items: center;
	gap: 6px;
	padding: 4px 10px;
	min-height: 28px;
	box-sizing: border-box;
	cursor: pointer;
	flex: 1;
	min-width: 0;
	background: ${(props) =>
		props.$isActive
			? `linear-gradient(var(--vscode-list-activeSelectionBackground), var(--vscode-list-activeSelectionBackground)), ${CODE_BLOCK_BG_COLOR}`
			: "transparent"};
	border-bottom: 2px solid ${(props) => (props.$isActive ? "var(--vscode-focusBorder)" : "transparent")};
	&:hover {
		background: var(--vscode-list-hoverBackground);
	}
`

const SplitModeLabel = styled.span<{ $mode: "plan" | "act" }>`
	font-size: 9px;
	font-weight: 600;
	color: ${(props) => (props.$mode === "plan" ? PLAN_MODE_COLOR : ACT_MODE_COLOR)};
	text-transform: uppercase;
`

const SplitModeModel = styled.span`
	font-size: 10px;
	color: var(--vscode-foreground);
	flex: 1;
	white-space: nowrap;
	overflow: hidden;
	text-overflow: ellipsis;
`

// Settings-only provider container with configured model info
const SettingsOnlyContainer = styled.div`
	display: flex;
	flex-direction: column;
	gap: 6px;
	padding: 12px 10px;
`

const ConfiguredModelName = styled.span`
	font-size: 11px;
	color: var(--vscode-foreground);
	font-weight: 500;
	overflow: hidden;
	text-overflow: ellipsis;
	white-space: nowrap;
`

const SettingsOnlyLink = styled.div`
	display: flex;
	align-items: center;
	justify-content: center;
	gap: 6px;
	padding: 6px 0;
	margin-top: 4px;
	cursor: pointer;
	color: var(--vscode-textLink-foreground);
	font-size: 11px;
	&:hover {
		text-decoration: underline;
	}
`

export default ModelPickerModal
