/**
 * @license
 * Copyright 2025 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import { logger } from '../logger';

/**
 * Defines the image format for images generated by Imagen.
 *
 * Use this class to specify the desired format (JPEG or PNG) and compression quality
 * for images generated by Imagen. This is typically included as part of
 * {@link ImagenModelParams}.
 *
 * @example
 * ```javascript
 * const imagenModelParams = {
 *   // ... other ImagenModelParams
 *   imageFormat: ImagenImageFormat.jpeg(75) // JPEG with a compression level of 75.
 * }
 * ```
 *
 * @public
 */
export class ImagenImageFormat {
  /**
   * The MIME type.
   */
  mimeType: string;
  /**
   * The level of compression (a number between 0 and 100).
   */
  compressionQuality?: number;

  private constructor() {
    this.mimeType = 'image/png';
  }

  /**
   * Creates an {@link ImagenImageFormat} for a JPEG image.
   *
   * @param compressionQuality - The level of compression (a number between 0 and 100).
   * @returns An {@link ImagenImageFormat} object for a JPEG image.
   *
   * @public
   */
  static jpeg(compressionQuality?: number): ImagenImageFormat {
    if (
      compressionQuality &&
      (compressionQuality < 0 || compressionQuality > 100)
    ) {
      logger.warn(
        `Invalid JPEG compression quality of ${compressionQuality} specified; the supported range is [0, 100].`
      );
    }
    return { mimeType: 'image/jpeg', compressionQuality };
  }

  /**
   * Creates an {@link ImagenImageFormat} for a PNG image.
   *
   * @returns An {@link ImagenImageFormat} object for a PNG image.
   *
   * @public
   */
  static png(): ImagenImageFormat {
    return { mimeType: 'image/png' };
  }
}
