/**
 * @license
 * Copyright 2025 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* eslint-disable @typescript-eslint/no-require-imports */
const { readdirSync, readFileSync, writeFileSync } = require('node:fs');
const { join } = require('node:path');

type BackendName = import('./types').BackendName; // Import type without triggering ES module detection

const MOCK_RESPONSES_DIR_PATH = join(
  __dirname,
  'vertexai-sdk-test-data',
  'mock-responses'
);
const MOCK_LOOKUP_OUTPUT_PATH = join(__dirname, 'mocks-lookup.ts');

const mockDirs: Record<BackendName, string> = {
  vertexAI: join(MOCK_RESPONSES_DIR_PATH, 'vertexai'),
  googleAI: join(MOCK_RESPONSES_DIR_PATH, 'googleai')
};

/**
 * Generates a JS file that exports maps from filenames to JSON mock responses (as strings)
 * for each backend.
 *
 * This allows tests that run in a browser to access the mock responses without having to
 * read from local disk and requiring 'fs'.
 */
function generateMockLookupFile(): void {
  console.log('Generating mock lookup file...');
  const vertexAIMockLookupText = generateMockLookup('vertexAI');
  const googleAIMockLookupText = generateMockLookup('googleAI');

  const fileText = `
/**
 * DO NOT EDIT - This file was generated by the packages/ai/test-utils/convert-mocks.ts script.
 * 
 * These objects map mock response filenames to their JSON contents.
 * 
 * Mock response files are pulled from https://github.com/FirebaseExtended/vertexai-sdk-test-data.
 */

// Automatically generated at: ${new Date().toISOString()}

${vertexAIMockLookupText}

${googleAIMockLookupText}
`;
  try {
    writeFileSync(MOCK_LOOKUP_OUTPUT_PATH, fileText, 'utf-8');
    console.log(
      `Successfully generated mock lookup file at: ${MOCK_LOOKUP_OUTPUT_PATH}`
    );
  } catch (err) {
    console.error(
      `Error writing mock lookup file to ${MOCK_LOOKUP_OUTPUT_PATH}:`,
      err
    );
    process.exit(1);
  }
}

/**
 * Given a directory that contains mock response files, reads through all the files,
 * maps file names to file contents, and returns a string of typescript code
 * that exports that map as an object.
 */
function generateMockLookup(backendName: BackendName): string {
  const lookup: Record<string, string> = {};
  const mockDir = mockDirs[backendName];
  let mockFilenames: string[];

  console.log(
    `Processing mocks for "${backendName}" from directory: ${mockDir}`
  );

  try {
    mockFilenames = readdirSync(mockDir);
  } catch (err) {
    console.error(
      `Error reading directory ${mockDir} for ${backendName}:`,
      err
    );
    return `export const ${backendName}MocksLookup: Record<string, string> = {};`;
  }

  if (mockFilenames.length === 0) {
    console.warn(`No .json files found in ${mockDir} for ${backendName}.`);
  }

  for (const mockFilename of mockFilenames) {
    const mockFilepath = `${mockDir}/${mockFilename}`;
    try {
      const fullText = readFileSync(mockFilepath, 'utf-8');
      lookup[mockFilename] = fullText;
    } catch (err) {
      console.error(
        `Error reading mock file ${mockFilepath} for ${backendName}:`,
        err
      );
    }
  }

  // Use JSON.stringify with indentation for readable output in the generated file
  const lookupJsonString = JSON.stringify(lookup, null, 2);

  return `export const ${backendName}MocksLookup: Record<string, string> = ${lookupJsonString};`;
}

generateMockLookupFile();
