/**
 * @license
 * Copyright 2020 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import {
  Endpoint,
  HttpMethod,
  RecaptchaClientType,
  RecaptchaVersion,
  _addTidIfNecessary,
  _makeTaggedError,
  _performApiRequest,
  _performSignInRequest
} from '../index';
import { AuthErrorCode } from '../../core/errors';
import { IdTokenResponse } from '../../model/id_token';
import { ServerError, ServerErrorMap } from '../errors';
import { Auth } from '../../model/public_types';

export interface SendPhoneVerificationCodeRequest {
  phoneNumber: string;
  // reCAPTCHA v2 token
  recaptchaToken?: string;
  tenantId?: string;
  // reCAPTCHA Enterprise token
  captchaResponse?: string;
  clientType?: RecaptchaClientType;
  recaptchaVersion?: RecaptchaVersion;
}

export interface SendPhoneVerificationCodeResponse {
  sessionInfo: string;
}

export async function sendPhoneVerificationCode(
  auth: Auth,
  request: SendPhoneVerificationCodeRequest
): Promise<SendPhoneVerificationCodeResponse> {
  return _performApiRequest<
    SendPhoneVerificationCodeRequest,
    SendPhoneVerificationCodeResponse
  >(
    auth,
    HttpMethod.POST,
    Endpoint.SEND_VERIFICATION_CODE,
    _addTidIfNecessary(auth, request)
  );
}

/**
 * @internal
 */
export interface SignInWithPhoneNumberRequest {
  temporaryProof?: string;
  phoneNumber?: string;
  sessionInfo?: string;
  code?: string;
  tenantId?: string;
}

export interface LinkWithPhoneNumberRequest
  extends SignInWithPhoneNumberRequest {
  idToken: string;
}

/**
 * @internal
 */
export interface SignInWithPhoneNumberResponse extends IdTokenResponse {
  temporaryProof?: string;
  phoneNumber?: string;
}

export async function signInWithPhoneNumber(
  auth: Auth,
  request: SignInWithPhoneNumberRequest
): Promise<SignInWithPhoneNumberResponse> {
  return _performSignInRequest<
    SignInWithPhoneNumberRequest,
    SignInWithPhoneNumberResponse
  >(
    auth,
    HttpMethod.POST,
    Endpoint.SIGN_IN_WITH_PHONE_NUMBER,
    _addTidIfNecessary(auth, request)
  );
}

export async function linkWithPhoneNumber(
  auth: Auth,
  request: LinkWithPhoneNumberRequest
): Promise<SignInWithPhoneNumberResponse> {
  const response = await _performSignInRequest<
    LinkWithPhoneNumberRequest,
    SignInWithPhoneNumberResponse
  >(
    auth,
    HttpMethod.POST,
    Endpoint.SIGN_IN_WITH_PHONE_NUMBER,
    _addTidIfNecessary(auth, request)
  );
  if (response.temporaryProof) {
    throw _makeTaggedError(auth, AuthErrorCode.NEED_CONFIRMATION, response);
  }
  return response;
}

interface VerifyPhoneNumberForExistingRequest
  extends SignInWithPhoneNumberRequest {
  operation: 'REAUTH';
}

const VERIFY_PHONE_NUMBER_FOR_EXISTING_ERROR_MAP_: Partial<
  ServerErrorMap<ServerError>
> = {
  [ServerError.USER_NOT_FOUND]: AuthErrorCode.USER_DELETED
};

export async function verifyPhoneNumberForExisting(
  auth: Auth,
  request: SignInWithPhoneNumberRequest
): Promise<SignInWithPhoneNumberResponse> {
  const apiRequest: VerifyPhoneNumberForExistingRequest = {
    ...request,
    operation: 'REAUTH'
  };
  return _performSignInRequest<
    VerifyPhoneNumberForExistingRequest,
    SignInWithPhoneNumberResponse
  >(
    auth,
    HttpMethod.POST,
    Endpoint.SIGN_IN_WITH_PHONE_NUMBER,
    _addTidIfNecessary(auth, apiRequest),
    VERIFY_PHONE_NUMBER_FOR_EXISTING_ERROR_MAP_
  );
}
