/**
 * @license
 * Copyright 2020 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import { BundleLoader } from '../core/bundle_impl';
import { createBundleReaderSync } from '../core/firestore_client';
import { newQueryComparator } from '../core/query';
import { ChangeType, ViewSnapshot } from '../core/view_snapshot';
import { FieldPath } from '../lite-api/field_path';
import {
  DocumentData,
  PartialWithFieldValue,
  Query,
  queryEqual,
  SetOptions,
  WithFieldValue
} from '../lite-api/reference';
import { LiteUserDataWriter } from '../lite-api/reference_impl';
import {
  DocumentSnapshot as LiteDocumentSnapshot,
  fieldPathFromArgument,
  FirestoreDataConverter as LiteFirestoreDataConverter
} from '../lite-api/snapshot';
import { UntypedFirestoreDataConverter } from '../lite-api/user_data_reader';
import { AbstractUserDataWriter } from '../lite-api/user_data_writer';
import { fromBundledQuery } from '../local/local_serializer';
import { documentKeySet } from '../model/collections';
import { Document } from '../model/document';
import { DocumentKey } from '../model/document_key';
import { DocumentSet } from '../model/document_set';
import { ResourcePath } from '../model/path';
import { newSerializer } from '../platform/serializer';
import {
  buildQuerySnapshotJsonBundle,
  buildDocumentSnapshotJsonBundle
} from '../platform/snapshot_to_json';
import { fromDocument } from '../remote/serializer';
import { debugAssert, fail } from '../util/assert';
import { Code, FirestoreError } from '../util/error';
// API extractor fails importing 'property' unless we also explicitly import 'Property'.
// eslint-disable-next-line @typescript-eslint/no-unused-vars, unused-imports/no-unused-imports-ts
import { Property, property, validateJSON } from '../util/json_validation';
import { AutoId } from '../util/misc';

import { Firestore } from './database';
import { SnapshotListenOptions } from './reference_impl';

const NOT_SUPPORTED = 'NOT SUPPORTED';

/**
 * Converter used by `withConverter()` to transform user objects of type
 * `AppModelType` into Firestore data of type `DbModelType`.
 *
 * Using the converter allows you to specify generic type arguments when
 * storing and retrieving objects from Firestore.
 *
 * In this context, an "AppModel" is a class that is used in an application to
 * package together related information and functionality. Such a class could,
 * for example, have properties with complex, nested data types, properties used
 * for memoization, properties of types not supported by Firestore (such as
 * `symbol` and `bigint`), and helper functions that perform compound
 * operations. Such classes are not suitable and/or possible to store into a
 * Firestore database. Instead, instances of such classes need to be converted
 * to "plain old JavaScript objects" (POJOs) with exclusively primitive
 * properties, potentially nested inside other POJOs or arrays of POJOs. In this
 * context, this type is referred to as the "DbModel" and would be an object
 * suitable for persisting into Firestore. For convenience, applications can
 * implement `FirestoreDataConverter` and register the converter with Firestore
 * objects, such as `DocumentReference` or `Query`, to automatically convert
 * `AppModel` to `DbModel` when storing into Firestore, and convert `DbModel`
 * to `AppModel` when retrieving from Firestore.
 *
 * @example
 *
 * Simple Example
 *
 * ```typescript
 * const numberConverter = {
 *     toFirestore(value: WithFieldValue<number>) {
 *         return { value };
 *     },
 *     fromFirestore(snapshot: QueryDocumentSnapshot, options: SnapshotOptions) {
 *         return snapshot.data(options).value as number;
 *     }
 * };
 *
 * async function simpleDemo(db: Firestore): Promise<void> {
 *     const documentRef = doc(db, 'values/value123').withConverter(numberConverter);
 *
 *     // converters are used with `setDoc`, `addDoc`, and `getDoc`
 *     await setDoc(documentRef, 42);
 *     const snapshot1 = await getDoc(documentRef);
 *     assertEqual(snapshot1.data(), 42);
 *
 *     // converters are not used when writing data with `updateDoc`
 *     await updateDoc(documentRef, { value: 999 });
 *     const snapshot2 = await getDoc(documentRef);
 *     assertEqual(snapshot2.data(), 999);
 * }
 * ```
 *
 * Advanced Example
 *
 * ```typescript
 * // The Post class is a model that is used by our application.
 * // This class may have properties and methods that are specific
 * // to our application execution, which do not need to be persisted
 * // to Firestore.
 * class Post {
 *     constructor(
 *         readonly title: string,
 *         readonly author: string,
 *         readonly lastUpdatedMillis: number
 *     ) {}
 *     toString(): string {
 *         return `${this.title} by ${this.author}`;
 *     }
 * }
 *
 * // The PostDbModel represents how we want our posts to be stored
 * // in Firestore. This DbModel has different properties (`ttl`,
 * // `aut`, and `lut`) from the Post class we use in our application.
 * interface PostDbModel {
 *     ttl: string;
 *     aut: { firstName: string; lastName: string };
 *     lut: Timestamp;
 * }
 *
 * // The `PostConverter` implements `FirestoreDataConverter` and specifies
 * // how the Firestore SDK can convert `Post` objects to `PostDbModel`
 * // objects and vice versa.
 * class PostConverter implements FirestoreDataConverter<Post, PostDbModel> {
 *     toFirestore(post: WithFieldValue<Post>): WithFieldValue<PostDbModel> {
 *         return {
 *             ttl: post.title,
 *             aut: this._autFromAuthor(post.author),
 *             lut: this._lutFromLastUpdatedMillis(post.lastUpdatedMillis)
 *         };
 *     }
 *
 *     fromFirestore(snapshot: QueryDocumentSnapshot, options: SnapshotOptions): Post {
 *         const data = snapshot.data(options) as PostDbModel;
 *         const author = `${data.aut.firstName} ${data.aut.lastName}`;
 *         return new Post(data.ttl, author, data.lut.toMillis());
 *     }
 *
 *     _autFromAuthor(
 *         author: string | FieldValue
 *     ): { firstName: string; lastName: string } | FieldValue {
 *         if (typeof author !== 'string') {
 *             // `author` is a FieldValue, so just return it.
 *             return author;
 *         }
 *         const [firstName, lastName] = author.split(' ');
 *         return {firstName, lastName};
 *     }
 *
 *     _lutFromLastUpdatedMillis(
 *         lastUpdatedMillis: number | FieldValue
 *     ): Timestamp | FieldValue {
 *         if (typeof lastUpdatedMillis !== 'number') {
 *             // `lastUpdatedMillis` must be a FieldValue, so just return it.
 *             return lastUpdatedMillis;
 *         }
 *         return Timestamp.fromMillis(lastUpdatedMillis);
 *     }
 * }
 *
 * async function advancedDemo(db: Firestore): Promise<void> {
 *     // Create a `DocumentReference` with a `FirestoreDataConverter`.
 *     const documentRef = doc(db, 'posts/post123').withConverter(new PostConverter());
 *
 *     // The `data` argument specified to `setDoc()` is type checked by the
 *     // TypeScript compiler to be compatible with `Post`. Since the `data`
 *     // argument is typed as `WithFieldValue<Post>` rather than just `Post`,
 *     // this allows properties of the `data` argument to also be special
 *     // Firestore values that perform server-side mutations, such as
 *     // `arrayRemove()`, `deleteField()`, and `serverTimestamp()`.
 *     await setDoc(documentRef, {
 *         title: 'My Life',
 *         author: 'Foo Bar',
 *         lastUpdatedMillis: serverTimestamp()
 *     });
 *
 *     // The TypeScript compiler will fail to compile if the `data` argument to
 *     // `setDoc()` is _not_ compatible with `WithFieldValue<Post>`. This
 *     // type checking prevents the caller from specifying objects with incorrect
 *     // properties or property values.
 *     // @ts-expect-error "Argument of type { ttl: string; } is not assignable
 *     // to parameter of type WithFieldValue<Post>"
 *     await setDoc(documentRef, { ttl: 'The Title' });
 *
 *     // When retrieving a document with `getDoc()` the `DocumentSnapshot`
 *     // object's `data()` method returns a `Post`, rather than a generic object,
 *     // which would have been returned if the `DocumentReference` did _not_ have a
 *     // `FirestoreDataConverter` attached to it.
 *     const snapshot1: DocumentSnapshot<Post> = await getDoc(documentRef);
 *     const post1: Post = snapshot1.data()!;
 *     if (post1) {
 *         assertEqual(post1.title, 'My Life');
 *         assertEqual(post1.author, 'Foo Bar');
 *     }
 *
 *     // The `data` argument specified to `updateDoc()` is type checked by the
 *     // TypeScript compiler to be compatible with `PostDbModel`. Note that
 *     // unlike `setDoc()`, whose `data` argument must be compatible with `Post`,
 *     // the `data` argument to `updateDoc()` must be compatible with
 *     // `PostDbModel`. Similar to `setDoc()`, since the `data` argument is typed
 *     // as `WithFieldValue<PostDbModel>` rather than just `PostDbModel`, this
 *     // allows properties of the `data` argument to also be those special
 *     // Firestore values, like `arrayRemove()`, `deleteField()`, and
 *     // `serverTimestamp()`.
 *     await updateDoc(documentRef, {
 *         'aut.firstName': 'NewFirstName',
 *         lut: serverTimestamp()
 *     });
 *
 *     // The TypeScript compiler will fail to compile if the `data` argument to
 *     // `updateDoc()` is _not_ compatible with `WithFieldValue<PostDbModel>`.
 *     // This type checking prevents the caller from specifying objects with
 *     // incorrect properties or property values.
 *     // @ts-expect-error "Argument of type { title: string; } is not assignable
 *     // to parameter of type WithFieldValue<PostDbModel>"
 *     await updateDoc(documentRef, { title: 'New Title' });
 *     const snapshot2: DocumentSnapshot<Post> = await getDoc(documentRef);
 *     const post2: Post = snapshot2.data()!;
 *     if (post2) {
 *         assertEqual(post2.title, 'My Life');
 *         assertEqual(post2.author, 'NewFirstName Bar');
 *     }
 * }
 * ```
 */
export interface FirestoreDataConverter<
  AppModelType,
  DbModelType extends DocumentData = DocumentData
> extends LiteFirestoreDataConverter<AppModelType, DbModelType> {
  /**
   * Called by the Firestore SDK to convert a custom model object of type
   * `AppModelType` into a plain JavaScript object (suitable for writing
   * directly to the Firestore database) of type `DbModelType`. To use `set()`
   * with `merge` and `mergeFields`, `toFirestore()` must be defined with
   * `PartialWithFieldValue<AppModelType>`.
   *
   * The `WithFieldValue<T>` type extends `T` to also allow FieldValues such as
   * {@link (deleteField:1)} to be used as property values.
   */
  toFirestore(
    modelObject: WithFieldValue<AppModelType>
  ): WithFieldValue<DbModelType>;

  /**
   * Called by the Firestore SDK to convert a custom model object of type
   * `AppModelType` into a plain JavaScript object (suitable for writing
   * directly to the Firestore database) of type `DbModelType`. Used with
   * {@link (setDoc:1)}, {@link (WriteBatch.set:1)} and
   * {@link (Transaction.set:1)} with `merge:true` or `mergeFields`.
   *
   * The `PartialWithFieldValue<T>` type extends `Partial<T>` to allow
   * FieldValues such as {@link (arrayUnion:1)} to be used as property values.
   * It also supports nested `Partial` by allowing nested fields to be
   * omitted.
   */
  toFirestore(
    modelObject: PartialWithFieldValue<AppModelType>,
    options: SetOptions
  ): PartialWithFieldValue<DbModelType>;

  /**
   * Called by the Firestore SDK to convert Firestore data into an object of
   * type `AppModelType`. You can access your data by calling:
   * `snapshot.data(options)`.
   *
   * Generally, the data returned from `snapshot.data()` can be cast to
   * `DbModelType`; however, this is not guaranteed because Firestore does not
   * enforce a schema on the database. For example, writes from a previous
   * version of the application or writes from another client that did not use a
   * type converter could have written data with different properties and/or
   * property types. The implementation will need to choose whether to
   * gracefully recover from non-conforming data or throw an error.
   *
   * To override this method, see {@link (FirestoreDataConverter.fromFirestore:1)}.
   *
   * @param snapshot - A `QueryDocumentSnapshot` containing your data and metadata.
   * @param options - The `SnapshotOptions` from the initial call to `data()`.
   */
  fromFirestore(
    snapshot: QueryDocumentSnapshot<DocumentData, DocumentData>,
    options?: SnapshotOptions
  ): AppModelType;
}

/**
 * Options that configure how data is retrieved from a `DocumentSnapshot` (for
 * example the desired behavior for server timestamps that have not yet been set
 * to their final value).
 */
export interface SnapshotOptions {
  /**
   * If set, controls the return value for server timestamps that have not yet
   * been set to their final value.
   *
   * By specifying 'estimate', pending server timestamps return an estimate
   * based on the local clock. This estimate will differ from the final value
   * and cause these values to change once the server result becomes available.
   *
   * By specifying 'previous', pending timestamps will be ignored and return
   * their previous value instead.
   *
   * If omitted or set to 'none', `null` will be returned by default until the
   * server value becomes available.
   */
  readonly serverTimestamps?: 'estimate' | 'previous' | 'none';
}

/**
 * Metadata about a snapshot, describing the state of the snapshot.
 */
export class SnapshotMetadata {
  /**
   * True if the snapshot contains the result of local writes (for example
   * `set()` or `update()` calls) that have not yet been committed to the
   * backend. If your listener has opted into metadata updates (via
   * `SnapshotListenOptions`) you will receive another snapshot with
   * `hasPendingWrites` equal to false once the writes have been committed to
   * the backend.
   */
  readonly hasPendingWrites: boolean;

  /**
   * True if the snapshot was created from cached data rather than guaranteed
   * up-to-date server data. If your listener has opted into metadata updates
   * (via `SnapshotListenOptions`) you will receive another snapshot with
   * `fromCache` set to false once the client has received up-to-date data from
   * the backend.
   */
  readonly fromCache: boolean;

  /** @hideconstructor */
  constructor(hasPendingWrites: boolean, fromCache: boolean) {
    this.hasPendingWrites = hasPendingWrites;
    this.fromCache = fromCache;
  }

  /**
   * Returns true if this `SnapshotMetadata` is equal to the provided one.
   *
   * @param other - The `SnapshotMetadata` to compare against.
   * @returns true if this `SnapshotMetadata` is equal to the provided one.
   */
  isEqual(other: SnapshotMetadata): boolean {
    return (
      this.hasPendingWrites === other.hasPendingWrites &&
      this.fromCache === other.fromCache
    );
  }
}

/**
 * The type of a `DocumentChange` may be 'added', 'removed', or 'modified'.
 */
export type DocumentChangeType = 'added' | 'removed' | 'modified';

/**
 * A `DocumentChange` represents a change to the documents matching a query.
 * It contains the document affected and the type of change that occurred.
 */
export interface DocumentChange<
  AppModelType = DocumentData,
  DbModelType extends DocumentData = DocumentData
> {
  /** The type of change ('added', 'modified', or 'removed'). */
  readonly type: DocumentChangeType;

  /** The document affected by this change. */
  readonly doc: QueryDocumentSnapshot<AppModelType, DbModelType>;

  /**
   * The index of the changed document in the result set immediately prior to
   * this `DocumentChange` (i.e. supposing that all prior `DocumentChange` objects
   * have been applied). Is `-1` for 'added' events.
   */
  readonly oldIndex: number;

  /**
   * The index of the changed document in the result set immediately after
   * this `DocumentChange` (i.e. supposing that all prior `DocumentChange`
   * objects and the current `DocumentChange` object have been applied).
   * Is -1 for 'removed' events.
   */
  readonly newIndex: number;
}

/**
 * A `DocumentSnapshot` contains data read from a document in your Firestore
 * database. The data can be extracted with `.data()` or `.get(<field>)` to
 * get a specific field.
 *
 * For a `DocumentSnapshot` that points to a non-existing document, any data
 * access will return 'undefined'. You can use the `exists()` method to
 * explicitly verify a document's existence.
 */
export class DocumentSnapshot<
  AppModelType = DocumentData,
  DbModelType extends DocumentData = DocumentData
> extends LiteDocumentSnapshot<AppModelType, DbModelType> {
  private readonly _firestoreImpl: Firestore;

  /**
   *  Metadata about the `DocumentSnapshot`, including information about its
   *  source and local modifications.
   */
  readonly metadata: SnapshotMetadata;

  /** @hideconstructor protected */
  constructor(
    readonly _firestore: Firestore,
    userDataWriter: AbstractUserDataWriter,
    key: DocumentKey,
    document: Document | null,
    metadata: SnapshotMetadata,
    converter: UntypedFirestoreDataConverter<AppModelType, DbModelType> | null
  ) {
    super(_firestore, userDataWriter, key, document, converter);
    this._firestoreImpl = _firestore;
    this.metadata = metadata;
  }

  /**
   * Returns whether or not the data exists. True if the document exists.
   */
  exists(): this is QueryDocumentSnapshot<AppModelType, DbModelType> {
    return super.exists();
  }

  /**
   * Retrieves all fields in the document as an `Object`. Returns `undefined` if
   * the document doesn't exist.
   *
   * By default, `serverTimestamp()` values that have not yet been
   * set to their final value will be returned as `null`. You can override
   * this by passing an options object.
   *
   * @param options - An options object to configure how data is retrieved from
   * the snapshot (for example the desired behavior for server timestamps that
   * have not yet been set to their final value).
   * @returns An `Object` containing all fields in the document or `undefined` if
   * the document doesn't exist.
   */
  data(options: SnapshotOptions = {}): AppModelType | undefined {
    if (!this._document) {
      return undefined;
    } else if (this._converter) {
      // We only want to use the converter and create a new DocumentSnapshot
      // if a converter has been provided.
      const snapshot = new QueryDocumentSnapshot(
        this._firestore,
        this._userDataWriter,
        this._key,
        this._document,
        this.metadata,
        /* converter= */ null
      );
      return this._converter.fromFirestore(snapshot, options);
    } else {
      return this._userDataWriter.convertValue(
        this._document.data.value,
        options.serverTimestamps
      ) as AppModelType;
    }
  }

  /**
   * Retrieves the field specified by `fieldPath`. Returns `undefined` if the
   * document or field doesn't exist.
   *
   * By default, a `serverTimestamp()` that has not yet been set to
   * its final value will be returned as `null`. You can override this by
   * passing an options object.
   *
   * @param fieldPath - The path (for example 'foo' or 'foo.bar') to a specific
   * field.
   * @param options - An options object to configure how the field is retrieved
   * from the snapshot (for example the desired behavior for server timestamps
   * that have not yet been set to their final value).
   * @returns The data at the specified field location or undefined if no such
   * field exists in the document.
   */
  // We are using `any` here to avoid an explicit cast by our users.
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  get(fieldPath: string | FieldPath, options: SnapshotOptions = {}): any {
    if (this._document) {
      const value = this._document.data.field(
        fieldPathFromArgument('DocumentSnapshot.get', fieldPath)
      );
      if (value !== null) {
        return this._userDataWriter.convertValue(
          value,
          options.serverTimestamps
        );
      }
    }
    return undefined;
  }

  static _jsonSchemaVersion: string = 'firestore/documentSnapshot/1.0';
  static _jsonSchema = {
    type: property('string', DocumentSnapshot._jsonSchemaVersion),
    bundleSource: property('string', 'DocumentSnapshot'),
    bundleName: property('string'),
    bundle: property('string')
  };

  /**
   * Returns a JSON-serializable representation of this `DocumentSnapshot` instance.
   *
   * @returns a JSON representation of this object.  Throws a {@link FirestoreError} if this
   * `DocumentSnapshot` has pending writes.
   */
  toJSON(): object {
    if (this.metadata.hasPendingWrites) {
      throw new FirestoreError(
        Code.FAILED_PRECONDITION,
        'DocumentSnapshot.toJSON() attempted to serialize a document with pending writes. ' +
          'Await waitForPendingWrites() before invoking toJSON().'
      );
    }
    const document = this._document;
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const result: any = {};
    result['type'] = DocumentSnapshot._jsonSchemaVersion;
    result['bundle'] = '';
    result['bundleSource'] = 'DocumentSnapshot';
    result['bundleName'] = this._key.toString();

    if (
      !document ||
      !document.isValidDocument() ||
      !document.isFoundDocument()
    ) {
      return result;
    }
    const documentData = this._userDataWriter.convertObjectMap(
      document.data.value.mapValue.fields,
      'previous'
    );
    result['bundle'] = buildDocumentSnapshotJsonBundle(
      this._firestore,
      document,
      documentData,
      this.ref.path
    );
    return result;
  }
}

/**
 * Builds a `DocumentSnapshot` instance from a JSON object created by
 * {@link DocumentSnapshot.toJSON}.
 *
 * @param firestore - The {@link Firestore} instance the snapshot should be loaded for.
 * @param json - a JSON object represention of a `DocumentSnapshot` instance.
 * @returns an instance of {@link DocumentSnapshot} if the JSON object could be
 * parsed. Throws a {@link FirestoreError} if an error occurs.
 */
export function documentSnapshotFromJSON(
  db: Firestore,
  json: object
): DocumentSnapshot;
/**
 * Builds a `DocumentSnapshot` instance from a JSON object created by
 * {@link DocumentSnapshot.toJSON}.
 *
 * @param firestore - The {@link Firestore} instance the snapshot should be loaded for.
 * @param json - a JSON object represention of a `DocumentSnapshot` instance.
 * @param converter - Converts objects to and from Firestore.
 * @returns an instance of {@link DocumentSnapshot} if the JSON object could be
 * parsed. Throws a {@link FirestoreError} if an error occurs.
 */
export function documentSnapshotFromJSON<
  AppModelType,
  DbModelType extends DocumentData = DocumentData
>(
  db: Firestore,
  json: object,
  converter: FirestoreDataConverter<AppModelType, DbModelType>
): DocumentSnapshot<AppModelType, DbModelType>;
export function documentSnapshotFromJSON<
  AppModelType,
  DbModelType extends DocumentData = DocumentData
>(
  db: Firestore,
  json: object,
  converter?: FirestoreDataConverter<AppModelType, DbModelType>
): DocumentSnapshot<AppModelType, DbModelType> {
  if (validateJSON(json, DocumentSnapshot._jsonSchema)) {
    if (json.bundle === NOT_SUPPORTED) {
      throw new FirestoreError(
        Code.INVALID_ARGUMENT,
        'The provided JSON object was created in a client environment, which is not supported.'
      );
    }
    // Parse the bundle data.
    const serializer = newSerializer(db._databaseId);
    const bundleReader = createBundleReaderSync(json.bundle, serializer);
    const elements = bundleReader.getElements();
    const bundleLoader: BundleLoader = new BundleLoader(
      bundleReader.getMetadata(),
      serializer
    );
    for (const element of elements) {
      bundleLoader.addSizedElement(element);
    }

    // Ensure that we have the correct number of documents in the bundle.
    const bundledDocuments = bundleLoader.documents;
    if (bundledDocuments.length !== 1) {
      throw new FirestoreError(
        Code.INVALID_ARGUMENT,
        `Expected bundle data to contain 1 document, but it contains ${bundledDocuments.length} documents.`
      );
    }

    // Build out the internal document data.
    const document = fromDocument(serializer, bundledDocuments[0].document!);
    const documentKey = new DocumentKey(
      ResourcePath.fromString(json.bundleName)
    );

    // Return the external facing DocumentSnapshot.
    return new DocumentSnapshot(
      db,
      new LiteUserDataWriter(db),
      documentKey,
      document,
      new SnapshotMetadata(
        /* hasPendingWrites= */ false,
        /* fromCache= */ false
      ),
      converter ? converter : null
    );
  }
  throw new FirestoreError(
    Code.INTERNAL,
    'Unexpected error creating DocumentSnapshot from JSON.'
  );
}

/**
 * A `QueryDocumentSnapshot` contains data read from a document in your
 * Firestore database as part of a query. The document is guaranteed to exist
 * and its data can be extracted with `.data()` or `.get(<field>)` to get a
 * specific field.
 *
 * A `QueryDocumentSnapshot` offers the same API surface as a
 * `DocumentSnapshot`. Since query results contain only existing documents, the
 * `exists` property will always be true and `data()` will never return
 * 'undefined'.
 */
export class QueryDocumentSnapshot<
  AppModelType = DocumentData,
  DbModelType extends DocumentData = DocumentData
> extends DocumentSnapshot<AppModelType, DbModelType> {
  /**
   * Retrieves all fields in the document as an `Object`.
   *
   * By default, `serverTimestamp()` values that have not yet been
   * set to their final value will be returned as `null`. You can override
   * this by passing an options object.
   *
   * @override
   * @param options - An options object to configure how data is retrieved from
   * the snapshot (for example the desired behavior for server timestamps that
   * have not yet been set to their final value).
   * @returns An `Object` containing all fields in the document.
   */
  data(options: SnapshotOptions = {}): AppModelType {
    return super.data(options) as AppModelType;
  }
}

/**
 * A `QuerySnapshot` contains zero or more `DocumentSnapshot` objects
 * representing the results of a query. The documents can be accessed as an
 * array via the `docs` property or enumerated using the `forEach` method. The
 * number of documents can be determined via the `empty` and `size`
 * properties.
 */
export class QuerySnapshot<
  AppModelType = DocumentData,
  DbModelType extends DocumentData = DocumentData
> {
  /**
   * Metadata about this snapshot, concerning its source and if it has local
   * modifications.
   */
  readonly metadata: SnapshotMetadata;

  /**
   * The query on which you called `get` or `onSnapshot` in order to get this
   * `QuerySnapshot`.
   */
  readonly query: Query<AppModelType, DbModelType>;

  private _cachedChanges?: Array<DocumentChange<AppModelType, DbModelType>>;
  private _cachedChangesIncludeMetadataChanges?: boolean;

  /** @hideconstructor */
  constructor(
    readonly _firestore: Firestore,
    readonly _userDataWriter: AbstractUserDataWriter,
    query: Query<AppModelType, DbModelType>,
    readonly _snapshot: ViewSnapshot
  ) {
    this.metadata = new SnapshotMetadata(
      _snapshot.hasPendingWrites,
      _snapshot.fromCache
    );
    this.query = query;
  }

  /** An array of all the documents in the `QuerySnapshot`. */
  get docs(): Array<QueryDocumentSnapshot<AppModelType, DbModelType>> {
    const result: Array<QueryDocumentSnapshot<AppModelType, DbModelType>> = [];
    this.forEach(doc => result.push(doc));
    return result;
  }

  /** The number of documents in the `QuerySnapshot`. */
  get size(): number {
    return this._snapshot.docs.size;
  }

  /** True if there are no documents in the `QuerySnapshot`. */
  get empty(): boolean {
    return this.size === 0;
  }

  /**
   * Enumerates all of the documents in the `QuerySnapshot`.
   *
   * @param callback - A callback to be called with a `QueryDocumentSnapshot` for
   * each document in the snapshot.
   * @param thisArg - The `this` binding for the callback.
   */
  forEach(
    callback: (
      result: QueryDocumentSnapshot<AppModelType, DbModelType>
    ) => void,
    thisArg?: unknown
  ): void {
    this._snapshot.docs.forEach(doc => {
      callback.call(
        thisArg,
        new QueryDocumentSnapshot<AppModelType, DbModelType>(
          this._firestore,
          this._userDataWriter,
          doc.key,
          doc,
          new SnapshotMetadata(
            this._snapshot.mutatedKeys.has(doc.key),
            this._snapshot.fromCache
          ),
          this.query.converter
        )
      );
    });
  }

  /**
   * Returns an array of the documents changes since the last snapshot. If this
   * is the first snapshot, all documents will be in the list as 'added'
   * changes.
   *
   * @param options - `SnapshotListenOptions` that control whether metadata-only
   * changes (i.e. only `DocumentSnapshot.metadata` changed) should trigger
   * snapshot events.
   */
  docChanges(
    options: SnapshotListenOptions = {}
  ): Array<DocumentChange<AppModelType, DbModelType>> {
    const includeMetadataChanges = !!options.includeMetadataChanges;

    if (includeMetadataChanges && this._snapshot.excludesMetadataChanges) {
      throw new FirestoreError(
        Code.INVALID_ARGUMENT,
        'To include metadata changes with your document changes, you must ' +
          'also pass { includeMetadataChanges:true } to onSnapshot().'
      );
    }

    if (
      !this._cachedChanges ||
      this._cachedChangesIncludeMetadataChanges !== includeMetadataChanges
    ) {
      this._cachedChanges = changesFromSnapshot(this, includeMetadataChanges);
      this._cachedChangesIncludeMetadataChanges = includeMetadataChanges;
    }

    return this._cachedChanges;
  }

  static _jsonSchemaVersion: string = 'firestore/querySnapshot/1.0';
  static _jsonSchema = {
    type: property('string', QuerySnapshot._jsonSchemaVersion),
    bundleSource: property('string', 'QuerySnapshot'),
    bundleName: property('string'),
    bundle: property('string')
  };

  /**
   * Returns a JSON-serializable representation of this `QuerySnapshot` instance.
   *
   * @returns a JSON representation of this object. Throws a {@link FirestoreError} if this
   * `QuerySnapshot` has pending writes.
   */
  toJSON(): object {
    if (this.metadata.hasPendingWrites) {
      throw new FirestoreError(
        Code.FAILED_PRECONDITION,
        'QuerySnapshot.toJSON() attempted to serialize a document with pending writes. ' +
          'Await waitForPendingWrites() before invoking toJSON().'
      );
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const result: any = {};
    result['type'] = QuerySnapshot._jsonSchemaVersion;
    result['bundleSource'] = 'QuerySnapshot';
    result['bundleName'] = AutoId.newId();

    const databaseId = this._firestore._databaseId.database;
    const projectId = this._firestore._databaseId.projectId;
    const parent = `projects/${projectId}/databases/${databaseId}/documents`;
    const documents: Document[] = [];
    const documentData: DocumentData[] = [];
    const paths: string[] = [];

    this.docs.forEach(doc => {
      if (doc._document === null) {
        return;
      }
      documents.push(doc._document);
      documentData.push(
        this._userDataWriter.convertObjectMap(
          doc._document.data.value.mapValue.fields,
          'previous'
        )
      );
      paths.push(doc.ref.path);
    });
    result['bundle'] = buildQuerySnapshotJsonBundle(
      this._firestore,
      this.query._query,
      result['bundleName'],
      parent,
      paths,
      documents,
      documentData
    );
    return result;
  }
}

/**
 * Builds a `QuerySnapshot` instance from a JSON object created by
 * {@link QuerySnapshot.toJSON}.
 *
 * @param firestore - The {@link Firestore} instance the snapshot should be loaded for.
 * @param json - a JSON object represention of a `QuerySnapshot` instance.
 * @returns an instance of {@link QuerySnapshot} if the JSON object could be
 * parsed. Throws a {@link FirestoreError} if an error occurs.
 */
export function querySnapshotFromJSON(
  db: Firestore,
  json: object
): QuerySnapshot;
/**
 * Builds a `QuerySnapshot` instance from a JSON object created by
 * {@link QuerySnapshot.toJSON}.
 *
 * @param firestore - The {@link Firestore} instance the snapshot should be loaded for.
 * @param json - a JSON object represention of a `QuerySnapshot` instance.
 * @param converter - Converts objects to and from Firestore.
 * @returns an instance of {@link QuerySnapshot} if the JSON object could be
 * parsed. Throws a {@link FirestoreError} if an error occurs.
 */
export function querySnapshotFromJSON<
  AppModelType,
  DbModelType extends DocumentData = DocumentData
>(
  db: Firestore,
  json: object,
  converter: FirestoreDataConverter<AppModelType, DbModelType>
): QuerySnapshot<AppModelType, DbModelType>;
export function querySnapshotFromJSON<
  AppModelType,
  DbModelType extends DocumentData
>(
  db: Firestore,
  json: object,
  converter?: FirestoreDataConverter<AppModelType, DbModelType>
): QuerySnapshot<AppModelType, DbModelType> {
  if (validateJSON(json, QuerySnapshot._jsonSchema)) {
    if (json.bundle === NOT_SUPPORTED) {
      throw new FirestoreError(
        Code.INVALID_ARGUMENT,
        'The provided JSON object was created in a client environment, which is not supported.'
      );
    }
    // Parse the bundle data.
    const serializer = newSerializer(db._databaseId);
    const bundleReader = createBundleReaderSync(json.bundle, serializer);
    const elements = bundleReader.getElements();
    const bundleLoader: BundleLoader = new BundleLoader(
      bundleReader.getMetadata(),
      serializer
    );
    for (const element of elements) {
      bundleLoader.addSizedElement(element);
    }

    if (bundleLoader.queries.length !== 1) {
      throw new FirestoreError(
        Code.INVALID_ARGUMENT,
        `Snapshot data expected 1 query but found ${bundleLoader.queries.length} queries.`
      );
    }

    // Create an internal Query object from the named query in the bundle.
    const query = fromBundledQuery(bundleLoader.queries[0].bundledQuery!);

    // Construct the arrays of document data for the query.
    const bundledDocuments = bundleLoader.documents;
    let documentSet = new DocumentSet();
    bundledDocuments.map(bundledDocument => {
      const document = fromDocument(serializer, bundledDocument.document!);
      documentSet = documentSet.add(document);
    });
    // Create a view snapshot of the query and documents.
    const viewSnapshot = ViewSnapshot.fromInitialDocuments(
      query,
      documentSet,
      documentKeySet() /* Zero mutated keys signifies no pending writes. */,
      /* fromCache= */ false,
      /* hasCachedResults= */ false
    );

    // Create an external Query object, required to construct the QuerySnapshot.
    const externalQuery = new Query<AppModelType, DbModelType>(
      db,
      converter ? converter : null,
      query
    );

    // Return a new QuerySnapshot with all of the collected data.
    return new QuerySnapshot<AppModelType, DbModelType>(
      db,
      new LiteUserDataWriter(db),
      externalQuery,
      viewSnapshot
    );
  }
  throw new FirestoreError(
    Code.INTERNAL,
    'Unexpected error creating QuerySnapshot from JSON.'
  );
}

/** Calculates the array of `DocumentChange`s for a given `ViewSnapshot`. */
export function changesFromSnapshot<
  AppModelType,
  DbModelType extends DocumentData
>(
  querySnapshot: QuerySnapshot<AppModelType, DbModelType>,
  includeMetadataChanges: boolean
): Array<DocumentChange<AppModelType, DbModelType>> {
  if (querySnapshot._snapshot.oldDocs.isEmpty()) {
    // Special case the first snapshot because index calculation is easy and
    // fast
    let lastDoc: Document;
    let index = 0;
    return querySnapshot._snapshot.docChanges.map(change => {
      debugAssert(
        change.type === ChangeType.Added,
        'Invalid event type for first snapshot'
      );
      debugAssert(
        !lastDoc ||
          newQueryComparator(querySnapshot._snapshot.query)(
            lastDoc,
            change.doc
          ) < 0,
        'Got added events in wrong order'
      );
      const doc = new QueryDocumentSnapshot<AppModelType, DbModelType>(
        querySnapshot._firestore,
        querySnapshot._userDataWriter,
        change.doc.key,
        change.doc,
        new SnapshotMetadata(
          querySnapshot._snapshot.mutatedKeys.has(change.doc.key),
          querySnapshot._snapshot.fromCache
        ),
        querySnapshot.query.converter
      );
      lastDoc = change.doc;
      return {
        type: 'added' as DocumentChangeType,
        doc,
        oldIndex: -1,
        newIndex: index++
      };
    });
  } else {
    // A `DocumentSet` that is updated incrementally as changes are applied to use
    // to lookup the index of a document.
    let indexTracker = querySnapshot._snapshot.oldDocs;
    return querySnapshot._snapshot.docChanges
      .filter(
        change => includeMetadataChanges || change.type !== ChangeType.Metadata
      )
      .map(change => {
        const doc = new QueryDocumentSnapshot<AppModelType, DbModelType>(
          querySnapshot._firestore,
          querySnapshot._userDataWriter,
          change.doc.key,
          change.doc,
          new SnapshotMetadata(
            querySnapshot._snapshot.mutatedKeys.has(change.doc.key),
            querySnapshot._snapshot.fromCache
          ),
          querySnapshot.query.converter
        );
        let oldIndex = -1;
        let newIndex = -1;
        if (change.type !== ChangeType.Added) {
          oldIndex = indexTracker.indexOf(change.doc.key);
          debugAssert(oldIndex >= 0, 'Index for document not found');
          indexTracker = indexTracker.delete(change.doc.key);
        }
        if (change.type !== ChangeType.Removed) {
          indexTracker = indexTracker.add(change.doc);
          newIndex = indexTracker.indexOf(change.doc.key);
        }
        return {
          type: resultChangeType(change.type),
          doc,
          oldIndex,
          newIndex
        };
      });
  }
}

export function resultChangeType(type: ChangeType): DocumentChangeType {
  switch (type) {
    case ChangeType.Added:
      return 'added';
    case ChangeType.Modified:
    case ChangeType.Metadata:
      return 'modified';
    case ChangeType.Removed:
      return 'removed';
    default:
      return fail(0xf03d, 'Unknown change type', { type });
  }
}

// TODO(firestoreexp): Add tests for snapshotEqual with different snapshot
// metadata
/**
 * Returns true if the provided snapshots are equal.
 *
 * @param left - A snapshot to compare.
 * @param right - A snapshot to compare.
 * @returns true if the snapshots are equal.
 */
export function snapshotEqual<AppModelType, DbModelType extends DocumentData>(
  left:
    | DocumentSnapshot<AppModelType, DbModelType>
    | QuerySnapshot<AppModelType, DbModelType>,
  right:
    | DocumentSnapshot<AppModelType, DbModelType>
    | QuerySnapshot<AppModelType, DbModelType>
): boolean {
  if (left instanceof DocumentSnapshot && right instanceof DocumentSnapshot) {
    return (
      left._firestore === right._firestore &&
      left._key.isEqual(right._key) &&
      (left._document === null
        ? right._document === null
        : left._document.isEqual(right._document)) &&
      left._converter === right._converter
    );
  } else if (left instanceof QuerySnapshot && right instanceof QuerySnapshot) {
    return (
      left._firestore === right._firestore &&
      queryEqual(left.query, right.query) &&
      left.metadata.isEqual(right.metadata) &&
      left._snapshot.isEqual(right._snapshot)
    );
  }

  return false;
}
