/*
 * Copyright 2018 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.google.cloud.bigtable.data.v2.models;

import static com.google.common.truth.Truth.assertThat;

import com.google.bigtable.v2.MutateRowRequest;
import com.google.bigtable.v2.MutateRowsRequest;
import com.google.bigtable.v2.Mutation.SetCell;
import com.google.cloud.bigtable.data.v2.internal.NameUtil;
import com.google.cloud.bigtable.data.v2.internal.RequestContext;
import com.google.common.primitives.Longs;
import com.google.protobuf.ByteString;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

@RunWith(JUnit4.class)
public class RowMutationTest {
  private static final String PROJECT_ID = "fake-project";
  private static final String INSTANCE_ID = "fake-instance";
  private static final String TABLE_ID = "fake-table";
  private static final String AUTHORIZED_VIEW_ID = "fake-authorized-view";
  private static final String APP_PROFILE_ID = "fake-profile";
  private static final RequestContext REQUEST_CONTEXT =
      RequestContext.create(PROJECT_ID, INSTANCE_ID, APP_PROFILE_ID);
  private static final ByteString TEST_KEY = ByteString.copyFromUtf8("fake-key");

  @Test
  public void toProtoTest() {
    long timestampMin = System.currentTimeMillis() * 1_000;

    // Test RowMutation on a table.
    RowMutation rowMutation =
        RowMutation.create(TABLE_ID, TEST_KEY)
            .setCell("fake-family", "fake-qualifier", "fake-value");

    MutateRowRequest actualRowMutation = rowMutation.toProto(REQUEST_CONTEXT);
    com.google.common.collect.Range<Long> timestampRange =
        com.google.common.collect.Range.closed(timestampMin, System.currentTimeMillis() * 1_000);

    assertThat(actualRowMutation.getTableName())
        .isEqualTo(NameUtil.formatTableName(PROJECT_ID, INSTANCE_ID, TABLE_ID));
    assertThat(actualRowMutation.getAuthorizedViewName()).isEmpty();
    assertThat(actualRowMutation.getAppProfileId()).isEqualTo(APP_PROFILE_ID);
    assertThat(actualRowMutation.getMutationsList()).hasSize(1);
    assertThat(actualRowMutation.getMutations(0).getSetCell().getValue())
        .isEqualTo(ByteString.copyFromUtf8("fake-value"));
    assertThat(actualRowMutation.getMutations(0).getSetCell().getTimestampMicros())
        .isIn(timestampRange);

    // Test RowMutation on an authorized view.
    rowMutation =
        RowMutation.create(AuthorizedViewId.of(TABLE_ID, AUTHORIZED_VIEW_ID), TEST_KEY)
            .setCell("fake-family", "fake-qualifier", "fake-value");

    actualRowMutation = rowMutation.toProto(REQUEST_CONTEXT);
    timestampRange =
        com.google.common.collect.Range.closed(timestampMin, System.currentTimeMillis() * 1_000);

    assertThat(actualRowMutation.getTableName()).isEmpty();
    assertThat(actualRowMutation.getAuthorizedViewName())
        .isEqualTo(
            NameUtil.formatAuthorizedViewName(
                PROJECT_ID, INSTANCE_ID, TABLE_ID, AUTHORIZED_VIEW_ID));
    assertThat(actualRowMutation.getAppProfileId()).isEqualTo(APP_PROFILE_ID);
    assertThat(actualRowMutation.getMutationsList()).hasSize(1);
    assertThat(actualRowMutation.getMutations(0).getSetCell().getValue())
        .isEqualTo(ByteString.copyFromUtf8("fake-value"));
    assertThat(actualRowMutation.getMutations(0).getSetCell().getTimestampMicros())
        .isIn(timestampRange);
  }

  @Test
  public void toBulkProtoTest() {
    long timestampMin = System.currentTimeMillis() * 1_000;

    // Test RowMutation on a table.
    RowMutation rowMutation =
        RowMutation.create(TABLE_ID, TEST_KEY)
            .setCell("fake-family", "fake-qualifier", "fake-value");

    MutateRowsRequest actualRowMutation = rowMutation.toBulkProto(REQUEST_CONTEXT);

    com.google.common.collect.Range<Long> timestampRange =
        com.google.common.collect.Range.closed(timestampMin, System.currentTimeMillis() * 1_000);

    assertThat(actualRowMutation.getTableName())
        .isEqualTo(NameUtil.formatTableName(PROJECT_ID, INSTANCE_ID, TABLE_ID));
    assertThat(actualRowMutation.getAuthorizedViewName()).isEmpty();
    assertThat(actualRowMutation.getAppProfileId()).isEqualTo(APP_PROFILE_ID);
    assertThat(actualRowMutation.getEntriesList()).hasSize(1);
    assertThat(actualRowMutation.getEntries(0).getMutationsList()).hasSize(1);
    assertThat(actualRowMutation.getEntries(0).getMutations(0).getSetCell().getValue())
        .isEqualTo(ByteString.copyFromUtf8("fake-value"));

    assertThat(actualRowMutation.getEntries(0).getMutations(0).getSetCell().getTimestampMicros())
        .isIn(timestampRange);

    // Test RowMutation on an authorized view.
    rowMutation =
        RowMutation.create(AuthorizedViewId.of(TABLE_ID, AUTHORIZED_VIEW_ID), TEST_KEY)
            .setCell("fake-family", "fake-qualifier", "fake-value");

    actualRowMutation = rowMutation.toBulkProto(REQUEST_CONTEXT);

    timestampRange =
        com.google.common.collect.Range.closed(timestampMin, System.currentTimeMillis() * 1_000);

    assertThat(actualRowMutation.getTableName()).isEmpty();
    assertThat(actualRowMutation.getAuthorizedViewName())
        .isEqualTo(
            NameUtil.formatAuthorizedViewName(
                PROJECT_ID, INSTANCE_ID, TABLE_ID, AUTHORIZED_VIEW_ID));
    assertThat(actualRowMutation.getAppProfileId()).isEqualTo(APP_PROFILE_ID);
    assertThat(actualRowMutation.getEntriesList()).hasSize(1);
    assertThat(actualRowMutation.getEntries(0).getMutationsList()).hasSize(1);
    assertThat(actualRowMutation.getEntries(0).getMutations(0).getSetCell().getValue())
        .isEqualTo(ByteString.copyFromUtf8("fake-value"));

    assertThat(actualRowMutation.getEntries(0).getMutations(0).getSetCell().getTimestampMicros())
        .isIn(timestampRange);
  }

  @Test
  public void toProtoTestWithProvidedMutation() {
    Mutation mutation = Mutation.create().setCell("fake-family", "fake-qualifier", "fake-value");
    // Test RowMutation on a table.
    RowMutation rowMutation = RowMutation.create(TABLE_ID, TEST_KEY, mutation);

    MutateRowRequest actualRowMutation = rowMutation.toProto(REQUEST_CONTEXT);

    assertThat(actualRowMutation.getMutationsList()).isEqualTo(mutation.getMutations());

    // Test RowMutation on an authorized view.
    rowMutation =
        RowMutation.create(AuthorizedViewId.of(TABLE_ID, AUTHORIZED_VIEW_ID), TEST_KEY, mutation);

    actualRowMutation = rowMutation.toProto(REQUEST_CONTEXT);

    assertThat(actualRowMutation.getMutationsList()).isEqualTo(mutation.getMutations());
  }

  @Test
  public void serializationTest() throws IOException, ClassNotFoundException {
    // Test RowMutation on a table.
    RowMutation expected =
        RowMutation.create(TABLE_ID, TEST_KEY)
            .setCell("fake-family", "fake-qualifier", 10_000, "fake-value");

    ByteArrayOutputStream bos = new ByteArrayOutputStream();
    ObjectOutputStream oos = new ObjectOutputStream(bos);
    oos.writeObject(expected);
    oos.close();

    ObjectInputStream ois = new ObjectInputStream(new ByteArrayInputStream(bos.toByteArray()));

    RowMutation actual = (RowMutation) ois.readObject();
    assertThat(actual.toProto(REQUEST_CONTEXT)).isEqualTo(expected.toProto(REQUEST_CONTEXT));

    // Test RowMutation on an authorized view.
    expected =
        RowMutation.create(AuthorizedViewId.of(TABLE_ID, AUTHORIZED_VIEW_ID), TEST_KEY)
            .setCell("fake-family", "fake-qualifier", 10_000, "fake-value");

    bos = new ByteArrayOutputStream();
    oos = new ObjectOutputStream(bos);
    oos.writeObject(expected);
    oos.close();

    ois = new ObjectInputStream(new ByteArrayInputStream(bos.toByteArray()));

    actual = (RowMutation) ois.readObject();
    assertThat(actual.toProto(REQUEST_CONTEXT)).isEqualTo(expected.toProto(REQUEST_CONTEXT));
  }

  @Test
  public void testWithLongValue() {
    // Test RowMutation on a table.
    RowMutation rowMutation =
        RowMutation.create(TABLE_ID, TEST_KEY)
            .setCell("fake-family", "fake-qualifier", 100_000L)
            .setCell("fake-family", "fake-qualifier", 30_000L, 100_000L);

    MutateRowRequest actualRowMutation = rowMutation.toProto(REQUEST_CONTEXT);

    SetCell setCell = actualRowMutation.getMutations(0).getSetCell();
    assertThat(setCell.getFamilyName()).isEqualTo("fake-family");
    assertThat(setCell.getColumnQualifier().toStringUtf8()).isEqualTo("fake-qualifier");
    assertThat(setCell.getValue()).isEqualTo(ByteString.copyFrom(Longs.toByteArray(100_000L)));

    assertThat(actualRowMutation.getMutations(1).getSetCell())
        .isEqualTo(
            SetCell.newBuilder()
                .setFamilyName("fake-family")
                .setColumnQualifier(ByteString.copyFromUtf8("fake-qualifier"))
                .setTimestampMicros(30_000L)
                .setValue(ByteString.copyFrom(Longs.toByteArray(100_000L)))
                .build());

    // Test RowMutation on an authorized view.
    rowMutation =
        RowMutation.create(AuthorizedViewId.of(TABLE_ID, AUTHORIZED_VIEW_ID), TEST_KEY)
            .setCell("fake-family", "fake-qualifier", 100_000L)
            .setCell("fake-family", "fake-qualifier", 30_000L, 100_000L);

    actualRowMutation = rowMutation.toProto(REQUEST_CONTEXT);

    setCell = actualRowMutation.getMutations(0).getSetCell();
    assertThat(setCell.getFamilyName()).isEqualTo("fake-family");
    assertThat(setCell.getColumnQualifier().toStringUtf8()).isEqualTo("fake-qualifier");
    assertThat(setCell.getValue()).isEqualTo(ByteString.copyFrom(Longs.toByteArray(100_000L)));

    assertThat(actualRowMutation.getMutations(1).getSetCell())
        .isEqualTo(
            SetCell.newBuilder()
                .setFamilyName("fake-family")
                .setColumnQualifier(ByteString.copyFromUtf8("fake-qualifier"))
                .setTimestampMicros(30_000L)
                .setValue(ByteString.copyFrom(Longs.toByteArray(100_000L)))
                .build());
  }

  @Test
  public void fromProtoTest() {
    // Test RowMutation on a table.
    RowMutation rowMutation =
        RowMutation.create(TABLE_ID, TEST_KEY)
            .setCell("fake-family", "fake-qualifier-1", "fake-value")
            .setCell("fake-family", "fake-qualifier-2", 30_000L, "fake-value-2");

    MutateRowRequest protoRequest = rowMutation.toProto(REQUEST_CONTEXT);
    RowMutation actualRequest = RowMutation.fromProto(protoRequest);

    assertThat(actualRequest.toProto(REQUEST_CONTEXT)).isEqualTo(protoRequest);

    String projectId = "fresh-project";
    String instanceId = "fresh-instance";
    String appProfile = "fresh-app-profile";
    MutateRowRequest overriddenRequest =
        actualRequest.toProto(RequestContext.create(projectId, instanceId, appProfile));

    assertThat(overriddenRequest).isNotEqualTo(protoRequest);
    assertThat(overriddenRequest.getTableName())
        .matches(NameUtil.formatTableName(projectId, instanceId, TABLE_ID));
    assertThat(overriddenRequest.getAuthorizedViewName()).isEmpty();
    assertThat(overriddenRequest.getAppProfileId()).matches(appProfile);

    // Test RowMutation on an authorized view.
    rowMutation =
        RowMutation.create(AuthorizedViewId.of(TABLE_ID, AUTHORIZED_VIEW_ID), TEST_KEY)
            .setCell("fake-family", "fake-qualifier-1", "fake-value")
            .setCell("fake-family", "fake-qualifier-2", 30_000L, "fake-value-2");

    protoRequest = rowMutation.toProto(REQUEST_CONTEXT);
    actualRequest = RowMutation.fromProto(protoRequest);

    assertThat(actualRequest.toProto(REQUEST_CONTEXT)).isEqualTo(protoRequest);

    overriddenRequest =
        actualRequest.toProto(RequestContext.create(projectId, instanceId, appProfile));

    assertThat(overriddenRequest).isNotEqualTo(protoRequest);
    assertThat(overriddenRequest.getTableName()).isEmpty();
    assertThat(overriddenRequest.getAuthorizedViewName())
        .matches(
            NameUtil.formatAuthorizedViewName(projectId, instanceId, TABLE_ID, AUTHORIZED_VIEW_ID));
    assertThat(overriddenRequest.getAppProfileId()).matches(appProfile);
  }
}
